#region Using directives
using System;
using FTOptix.Core;
using UAManagedCore;
using OpcUa = UAManagedCore.OpcUa;
using FTOptix.HMIProject;
using FTOptix.OPCUAServer;
using FTOptix.UI;
using FTOptix.NativeUI;
using FTOptix.CoreBase;
using FTOptix.NetLogic;
using FTOptix.CommunicationDriver;
using FTOptix.SerialPort;
using System.Collections.Generic;
#endregion

public class RuntimeNetlogic1 : BaseNetLogic
{
    private SerialPort serialPort;
    private PeriodicTask periodicTask;
    private IUAVariable tag;
    private readonly int CRCSizeInBytes = 2;
    private IEventObserver observer;
    private IEventRegistration registration;

    public override void Start()
    {
        serialPort = (SerialPort)Owner;
        serialPort.Baudrate = 14400;
        tag = serialPort.Get<IUAVariable>("tag1");

        observer = new CallbackVariableChangeObserver(OnCommunicationStatusVariableValueChanged);
        registration = serialPort.CommunicationStatusVariable.RegisterEventObserver(observer, EventType.VariableValueChanged);

        periodicTask = new PeriodicTask(Read, 300, Owner);
        periodicTask.Start();
    }

    private void Read()
    {
        try
        {
            ReadImpl();
        }
        catch (Exception)
        {
        }
    }

    private void ReadImpl()
    {
        serialPort.WriteBytes(Serialize());
        var result = serialPort.ReadBytes(3);

        if ((result[1] & 0x80) == 0)
        {
            result = serialPort.ReadBytes((uint)(result[2] + CRCSizeInBytes));
            tag.Value = Deserialize(result);
        }
        else
        {
            Log.Error("Failed to read from Modbus");
        }
    }

    private byte[] Serialize()
    {
        var buffer = new List<byte>
        {
            0x01, // UnitId
            0x03, // Function code
            0x00, // Starting address
            0x00,
            0x00, // Quantity Of Registers
            0x01,
            0x84, // CRC
            0x0a
        };

        return buffer.ToArray();
    }

    private UInt16 Deserialize(byte[] buffer)
    {
        var first = (UInt16)buffer[1];
        var second = (UInt16)(buffer[0] << 8);
        return (UInt16)(first | second);
    }

    private void ReadASCIIImpl()
    {
        serialPort.WriteChars(SerializeASCII());
        var result = serialPort.ReadCharsUntil("\r\n");

        if ((result[1] & 0x80) == 0)
        {
            tag.Value = DeserializeASCII(result);
        }
        else
        {
            Log.Error("Failed to read from Modbus");
        }
    }

    private char[] SerializeASCII()
    {
        var buffer = new List<char>
        {
            ':' ,
            '0' ,
            '1' ,
            '0' ,
            '3' ,
            '0' ,
            '0' ,
            '0' ,
            '0' ,
            '0' ,
            '0' ,
            '0' ,
            '1' ,
            'F' ,
            'B' ,
            '\r',
            '\n',
        };

        return buffer.ToArray();
    }

    private UInt16 DeserializeASCII(char[] buffer)
    {
        var resultBuffer = new char[3];
        Array.Copy(buffer, 8, resultBuffer, 0, resultBuffer.Length);
        return UInt16.Parse(new string(resultBuffer), System.Globalization.NumberStyles.HexNumber);
    }

    private void OnCommunicationStatusVariableValueChanged(IUAVariable variable, UAValue newValue, UAValue oldValue, uint[] indexes, ulong senderId)
    {
        Log.Info("Communication status changed: " + newValue.Value.ToString());
    }
}
