#region Using directives
using System;
using UAManagedCore;
using OpcUa = UAManagedCore.OpcUa;
using FTOptix.Store;
using FTOptix.SQLiteStore;
using FTOptix.DataLogger;
using FTOptix.HMIProject;
using FTOptix.NetLogic;
using FTOptix.Core;
using CloudConnector;
using System.Security.Cryptography.X509Certificates;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Globalization;
using System.Text;
using System.IO;
using MQTTnet;
using MQTTnet.Client;
using MQTTnet.Exceptions;
using Newtonsoft.Json;
#endregion

namespace CloudConnector
{
    public abstract class Record
    {
        public Record(DateTime? timestamp)
        {
            this.timestamp = timestamp;
        }

        public override bool Equals(object obj)
        {
            var other = obj as Record;
            return timestamp == other.timestamp;
        }

        public readonly DateTime? timestamp;
    }

    public class DataLoggerRecord : Record
    {
        public DataLoggerRecord(DateTime timestamp, List<VariableRecord> variables) : base(timestamp)
        {
            this.variables = variables;
        }

        public DataLoggerRecord(DateTime timestamp, DateTime? localTimestamp, List<VariableRecord> variables) : base(timestamp)
        {
            this.localTimestamp = localTimestamp;
            this.variables = variables;
        }

        public override bool Equals(object obj)
        {
            DataLoggerRecord other = obj as DataLoggerRecord;

            if (other == null)
                return false;

            if (timestamp != other.timestamp)
                return false;

            if (localTimestamp != other.localTimestamp)
                return false;

            if (variables.Count != other.variables.Count)
                return false;

            for (int i = 0; i < variables.Count; ++i)
            {
                if (!variables[i].Equals(other.variables[i]))
                    return false;
            }

            return true;
        }

        public readonly DateTime? localTimestamp;
        public readonly List<VariableRecord> variables;
    }

    public class VariableRecord : Record
    {
        public VariableRecord(DateTime? timestamp,
                              string variableId,
                              UAValue value,
                              string serializedValue) : base(timestamp)
        {
            this.variableId = variableId;
            this.value = value;
            this.serializedValue = serializedValue;
            this.variableOpCode = null;
        }

        public VariableRecord(DateTime? timestamp,
                              string variableId,
                              UAValue value,
                              string serializedValue,
                              int? variableOpCode) : base(timestamp)
        {
            this.variableId = variableId;
            this.value = value;
            this.serializedValue = serializedValue;
            this.variableOpCode = variableOpCode;
        }

        public override bool Equals(object obj)
        {
            var other = obj as VariableRecord;
            return timestamp == other.timestamp &&
                   variableId == other.variableId &&
                   value == other.value &&
                   serializedValue == other.serializedValue &&
                   variableOpCode == other.variableOpCode;
        }

        public readonly string variableId;
        public readonly string serializedValue;
        public readonly UAValue value;
        public readonly int? variableOpCode;
    }

    public class Packet
    {
        public Packet(DateTime timestamp, string clientId)
        {
            this.timestamp = timestamp.ToUniversalTime();
            this.clientId = clientId;
        }

        public readonly DateTime timestamp;
        public readonly string clientId;
    }

    public class VariablePacket : Packet
    {
        public VariablePacket(DateTime timestamp,
                              string clientId,
                              List<VariableRecord> records) : base(timestamp, clientId)
        {
            this.records = records;
        }

        public readonly List<VariableRecord> records;
    }

    public class DataLoggerRowPacket : Packet
    {
        public DataLoggerRowPacket(DateTime timestamp,
                                   string clientId,
                                   List<DataLoggerRecord> records) : base(timestamp, clientId)
        {
            this.records = records;
        }

        public readonly List<DataLoggerRecord> records;
    }

    public class DataLoggerRecordUtils
    {
        public static List<DataLoggerRecord> GetDataLoggerRecordsFromQueryResult(object[,] resultSet,
                                                                                 string[] header,
                                                                                 List<VariableToLog> variablesToLogList,
                                                                                 bool insertOpCode,
                                                                                 bool insertVariableTimestamp,
                                                                                 bool logLocalTime)
        {
            var records = new List<DataLoggerRecord>();
            var rowCount = resultSet != null ? resultSet.GetLength(0) : 0;
            var columnCount = header != null ? header.Length : 0;
            for (int i = 0; i < rowCount; ++i)
            {
                var j = 0;
                var rowVariables = new List<VariableRecord>();
                DateTime rowTimestamp = GetTimestamp(resultSet[i, j++]);
                DateTime? rowLocalTimestamp = null;
                if (logLocalTime)
                    rowLocalTimestamp = DateTime.Parse(resultSet[i, j++].ToString());

                int variableIndex = 0;
                while (j < columnCount)
                {
                    string variableId = header[j];
                    object value = resultSet[i, j];
                    string serializedValue = SerializeValue(value, variablesToLogList[variableIndex]);

                    DateTime? timestamp = null;
                    if (insertVariableTimestamp)
                    {
                        ++j; // Consume timestamp column
                        var timestampColumnValue = resultSet[i, j];
                        if (timestampColumnValue != null)
                            timestamp = GetTimestamp(timestampColumnValue);
                    }

                    VariableRecord variableRecord;
                    if (insertOpCode)
                    {
                        ++j; // Consume operation code column
                        var opCodeColumnValue = resultSet[i, j];
                        int? opCode = (opCodeColumnValue != null) ? (Int32.Parse(resultSet[i, j].ToString())) : (int?)null;
                        variableRecord = new VariableRecord(timestamp, variableId, GetUAValue(value, variablesToLogList[variableIndex]), serializedValue, opCode);
                    }
                    else
                        variableRecord = new VariableRecord(timestamp, variableId, GetUAValue(value, variablesToLogList[variableIndex]), serializedValue);

                    rowVariables.Add(variableRecord);

                    ++j; // Consume Variable Column
                    ++variableIndex;
                }

                DataLoggerRecord record;
                if (logLocalTime)
                    record = new DataLoggerRecord(rowTimestamp, rowLocalTimestamp, rowVariables);
                else
                    record = new DataLoggerRecord(rowTimestamp, rowVariables);

                records.Add(record);
            }

            return records;
        }

        private static string SerializeValue(object value, VariableToLog variableToLog)
        {
            if (value == null)
                return null;
            var valueType = variableToLog.ActualDataType;
            if (valueType == OpcUa.DataTypes.DateTime)
                return (GetTimestamp(value)).ToString("O");
            else if (valueType == OpcUa.DataTypes.Float)
                return ((float)((double)value)).ToString("G9");
            else if (valueType == OpcUa.DataTypes.Double)
                return ((double)value).ToString("G17");

            return value.ToString();
        }

        private static UAValue GetUAValue(object value, VariableToLog variableToLog)
        {
            if (value == null)
                return null;
            try
            {
                NodeId valueType = variableToLog.ActualDataType;
                if (valueType == OpcUa.DataTypes.Boolean)
                    return new UAValue(Int32.Parse(GetBoolean(value)));
                else if (valueType == OpcUa.DataTypes.Integer)
                    return new UAValue(Int64.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.UInteger)
                    return new UAValue(UInt64.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.Byte)
                    return new UAValue(Byte.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.SByte)
                    return new UAValue(SByte.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.Int16)
                    return new UAValue(Int16.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.UInt16)
                    return new UAValue(UInt16.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.Int32)
                    return new UAValue(Int32.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.UInt32)
                    return new UAValue(UInt32.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.Int64)
                    return new UAValue(Int64.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.UInt64)
                    return new UAValue(UInt64.Parse(value.ToString()));
                else if (valueType == OpcUa.DataTypes.Float)
                    return new UAValue((float)((double)value));
                else if (valueType == OpcUa.DataTypes.Double)
                    return new UAValue((double)value);
                else if (valueType == OpcUa.DataTypes.DateTime)
                    return new UAValue(GetTimestamp(value));
                else if (valueType == OpcUa.DataTypes.String)
                    return new UAValue(value.ToString());
                else if (valueType == OpcUa.DataTypes.ByteString)
                    return new UAValue((ByteString)value);
                else if (valueType == OpcUa.DataTypes.NodeId)
                    return new UAValue((NodeId)value);
            }
            catch (Exception e)
            {
                Log.Warning("DataLoggerRecordUtils", $"Parse Exception: {e.Message}.");
                throw;
            }

            return null;
        }

        private static string GetBoolean(object value)
        {
            var valueString = value.ToString();
            if (valueString == "0" || valueString == "1")
                return valueString;

            if (valueString.ToLower() == "false")
                return "0";
            else
                return "1";
        }

        private static DateTime GetTimestamp(object value)
        {
            if (Type.GetTypeCode(value.GetType()) == TypeCode.DateTime)
                return ((DateTime)value);
            else
                return DateTime.SpecifyKind(DateTime.Parse(value.ToString()), DateTimeKind.Utc);
        }
    }

    public class DataLoggerStoreWrapper
    {
        public DataLoggerStoreWrapper(Store store,
                                      string tableName,
                                      List<VariableToLog> variablesToLogList,
                                      bool insertOpCode,
                                      bool insertVariableTimestamp,
                                      bool logLocalTime)
        {
            this.store = store;
            this.tableName = tableName;
            this.variablesToLogList = variablesToLogList;
            this.insertOpCode = insertOpCode;
            this.insertVariableTimestamp = insertVariableTimestamp;
            this.logLocalTime = logLocalTime;
        }

        public void DeletePulledRecords()
        {
            if (store.Status == StoreStatus.Offline)
                return;

            try
            {
                Log.Verbose1("DataLoggerStoreWrapper", "Delete records pulled from data logger temporary table.");

                string query = $"DELETE FROM \"{tableName}\" AS D " +
                               $"WHERE \"Id\" IN " +
                               $"( SELECT \"Id\" " +
                               $"FROM \"##tempDataLoggerTable\")";

                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to delete from data logger temporary table {e.Message}.");
                throw;
            }

            DeleteTemporaryTable();
        }

        public List<DataLoggerRecord> QueryNewEntries()
        {
            Log.Verbose1("DataLoggerStoreWrapper", "Query new entries from data logger.");

            if (store.Status == StoreStatus.Offline)
                return new List<DataLoggerRecord>();

            CopyNewEntriesToTemporaryTable();
            List<DataLoggerRecord> records = QueryNewEntriesFromTemporaryTable();

            if (records.Count == 0)
                DeleteTemporaryTable();

            return records;
        }

        public List<DataLoggerRecord> QueryNewEntriesUsingLastQueryId(UInt64 rowId)
        {
            Log.Verbose1("DataLoggerStoreWrapper", $"Query new entries with id greater than {rowId} (store status: {store.Status}).");

            if (store.Status == StoreStatus.Offline)
                return new List<DataLoggerRecord>();

            CopyNewEntriesToTemporaryTableUsingId(rowId);
            List<DataLoggerRecord> records = QueryNewEntriesFromTemporaryTable();

            if (records.Count == 0)
                DeleteTemporaryTable();

            return records;
        }

        public UInt64? GetMaxIdFromTemporaryTable()
        {
            object[,] resultSet;

            try
            {
                string query = $"SELECT MAX(\"Id\") FROM \"##tempDataLoggerTable\"";

                if (store.Status == StoreStatus.Online)
                {
                    store.Query(query, out _, out resultSet);
                    DeleteTemporaryTable();

                    if (resultSet[0, 0] != null)
                    {
                        Log.Verbose1("DataLoggerStoreWrapper", $"Get max id from data logger temporary table returns {resultSet[0, 0]}.");
                        return UInt64.Parse(resultSet[0, 0].ToString());
                    }
                }

                return null;
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to query maxid from data logger temporary table: {e.Message}.");
                throw;
            }
        }

        public UInt64? GetDataLoggerMaxId()
        {
            object[,] resultSet;

            try
            {
                string query = $"SELECT MAX(\"Id\") FROM \"{tableName}\"";

                if (store.Status == StoreStatus.Online)
                {
                    store.Query(query, out _, out resultSet);

                    if (resultSet[0, 0] != null)
                    {
                        Log.Verbose1("DataLoggerStoreWrapper", $"Get data logger max id returns {resultSet[0, 0]}.");
                        return UInt64.Parse(resultSet[0, 0].ToString());
                    }
                }

                return null;
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to query maxid from data logger temporary table: {e.Message}.");
                throw;
            }
        }

        public StoreStatus GetStoreStatus()
        {
            return store.Status;
        }

        private void CopyNewEntriesToTemporaryTable()
        {
            try
            {
                Log.Verbose1("DataLoggerStoreWrapper", "Copy new entries to data logger temporary table.");

                string query = $"CREATE TEMPORARY TABLE \"##tempDataLoggerTable\" AS " +
                               $"SELECT * " +
                               $"FROM \"{tableName}\" " +
                               $"WHERE \"Id\" IS NOT NULL " +
                               $"ORDER BY \"Timestamp\" ASC ";

                if (store.Status == StoreStatus.Online)
                    store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to create internal temporary table: {e.Message}.");
                throw;
            }
        }

        private void CopyNewEntriesToTemporaryTableUsingId(UInt64 rowId)
        {
            try
            {
                Int64 id = rowId == Int64.MaxValue ? -1 : (Int64)rowId; // -1 to consider also id = 0
                Log.Verbose1("DataLoggerStoreWrapper", $"Copy new entries to data logger temporary table with id greater than {id}.");

                string query = $"CREATE TEMPORARY TABLE \"##tempDataLoggerTable\" AS " +
                               $"SELECT * " +
                               $"FROM \"{tableName}\" " +
                               $"WHERE \"Id\" > {id} " +
                               $"ORDER BY \"Timestamp\" ASC ";

                if (store.Status == StoreStatus.Online)
                    store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to create internal temporary table: {e.Message}.");
                throw;
            }
        }

        private void DeleteTemporaryTable()
        {
            object[,] resultSet;
            string[] header;

            try
            {
                Log.Verbose1("DataLoggerStoreWrapper", "Delete data logger temporary table.");
                string query = $"DROP TABLE \"##tempDataLoggerTable\"";
                store.Query(query, out header, out resultSet);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to delete internal temporary table: {e.Message}.");
                throw;
            }
        }

        private List<DataLoggerRecord> QueryNewEntriesFromTemporaryTable()
        {
            List<DataLoggerRecord> records = null;
            object[,] resultSet;
            string[] header;

            try
            {
                string query = $"SELECT {GetQuerySelectParameters()} " +
                               $"FROM \"##tempDataLoggerTable\"";

                if (store.Status == StoreStatus.Online)
                {
                    store.Query(query, out header, out resultSet);
                    records = DataLoggerRecordUtils.GetDataLoggerRecordsFromQueryResult(resultSet,
                                                                                        header,
                                                                                        variablesToLogList,
                                                                                        insertOpCode,
                                                                                        insertVariableTimestamp,
                                                                                        logLocalTime);
                }
                else
                    records = new List<DataLoggerRecord>();

                Log.Verbose1("DataLoggerStoreWrapper", $"Query new entries from data logger temporary table (records count={records.Count}, query={query}).");
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStoreWrapper", $"Failed to query the internal temporary table: {e.Message}.");
                throw;
            }

            return records;
        }

        private string GetQuerySelectParameters()
        {
            var selectParameters = "\"Timestamp\", ";
            if (logLocalTime)
                selectParameters += "\"LocalTimestamp\", ";

            selectParameters = $"{selectParameters} {GetQueryColumnsOrderedByVariableName()}";

            return selectParameters;
        }

        private string GetQueryColumnsOrderedByVariableName()
        {
            var columnsOrderedByVariableName = string.Empty;
            foreach (var variable in variablesToLogList)
            {
                if (columnsOrderedByVariableName != string.Empty)
                    columnsOrderedByVariableName += ", ";

                columnsOrderedByVariableName += "\"" + variable.BrowseName + "\"";

                if (insertVariableTimestamp)
                    columnsOrderedByVariableName += ", \"" + variable.BrowseName + "_Timestamp\"";

                if (insertOpCode)
                    columnsOrderedByVariableName += ", \"" + variable.BrowseName + "_OpCode\"";
            }

            return columnsOrderedByVariableName;
        }

        private readonly Store store;
        private readonly string tableName;
        private readonly List<VariableToLog> variablesToLogList;
        private readonly bool insertOpCode;
        private readonly bool insertVariableTimestamp;
        private readonly bool logLocalTime;
    }

    public interface SupportStore
    {
        void InsertRecords(List<Record> records);
        void DeleteRecords(int numberOfRecordsToDelete);
        long RecordsCount();
        List<Record> QueryOlderEntries(int numberOfEntries);
    }

    public class PushAgentStoreDataLoggerWrapper : SupportStore
    {
        public PushAgentStoreDataLoggerWrapper(Store store,
                                               string tableName,
                                               List<VariableToLog> variablesToLogList,
                                               bool insertOpCode,
                                               bool insertVariableTimestamp,
                                               bool logLocalTime)
        {
            this.store = store;
            this.tableName = tableName;
            this.variablesToLogList = variablesToLogList;
            this.insertOpCode = insertOpCode;
            this.insertVariableTimestamp = insertVariableTimestamp;
            this.logLocalTime = logLocalTime;

            try
            {
                CreateTable();
                table = GetTable();
                CreateColumns();
                CreateColumnIndex("Id", true);
                CreateColumnIndex("Timestamp", false);
                columns = GetTableColumnsOrderedByVariableName();
                idCount = GetMaxId();
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Unable to create PushAgent store: {e.Message}.");
                throw;
            }
        }

        public void DeleteRecords(int numberOfRecordsToDelete)
        {
            try
            {

                string query = $"DELETE FROM \"{tableName}\" " +
                               $"ORDER BY \"Timestamp\" ASC, \"Id\" ASC " +
                               $"LIMIT {numberOfRecordsToDelete}";

                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Failed to delete data from PushAgent store: {e.Message}.");
                throw;
            }
        }

        public void InsertRecords(List<Record> records)
        {
            List<DataLoggerRecord> dataLoggerRecords = records.Cast<DataLoggerRecord>().ToList();
            object[,] values = new object[records.Count, columns.Length];
            ulong tempIdCount = idCount;
            for (int i = 0; i < dataLoggerRecords.Count; ++i)
            {
                int j = 0;
                values[i, j++] = tempIdCount;
                values[i, j++] = dataLoggerRecords[i].timestamp;
                if (logLocalTime)
                    values[i, j++] = dataLoggerRecords[i].localTimestamp;

                foreach (var variable in dataLoggerRecords.ElementAt(i).variables)
                {
                    values[i, j++] = variable.value?.Value;
                    if (insertVariableTimestamp)
                        values[i, j++] = variable.timestamp;
                    if (insertOpCode)
                        values[i, j++] = variable.variableOpCode;
                }

                tempIdCount = GetNextInternalId(tempIdCount);
            }

            try
            {
                table.Insert(columns, values);
                idCount = tempIdCount;          // If all record are inserted then we update the idCount
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Failed to insert data into PushAgent store: {e.Message}.");
                throw;
            }
        }

        public List<Record> QueryOlderEntries(int numberOfEntries)
        {
            List<Record> records = null;
            object[,] resultSet;
            string[] header;

            try
            {
                string query = $"SELECT {GetQuerySelectParameters()} " +
                               $"FROM \"{tableName}\" " +
                               $"ORDER BY \"Timestamp\" ASC, \"Id\" ASC " +
                               $"LIMIT {numberOfEntries}";

                store.Query(query, out header, out resultSet);
                records = DataLoggerRecordUtils.GetDataLoggerRecordsFromQueryResult(resultSet,
                                                                                    header,
                                                                                    variablesToLogList,
                                                                                    insertOpCode,
                                                                                    insertVariableTimestamp,
                                                                                    logLocalTime).Cast<Record>().ToList();
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Failed to query older entries from PushAgent store: {e.Message}.");
                throw;
            }

            return records;
        }

        public long RecordsCount()
        {
            object[,] resultSet;
            long result = 0;

            try
            {
                string query = $"SELECT COUNT(*) FROM \"{tableName}\"";
                store.Query(query, out _, out resultSet);
                result = ((long)resultSet[0, 0]);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Failed to query count: {e.Message}.");
                throw;
            }

            return result;
        }

        private UInt64 GetMaxId()
        {
            object[,] resultSet;

            try
            {
                string query = $"SELECT MAX(\"Id\") FROM \"{tableName}\"";
                store.Query(query, out _, out resultSet);

                if (resultSet[0, 0] != null)
                    return GetNextInternalId(UInt64.Parse(resultSet[0, 0].ToString()));
                else
                    return 0;
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Failed to query maxid: {e.Message}.");
                throw;
            }
        }

        private UInt64 GetNextInternalId(UInt64 currentId)
        {
            return currentId < Int64.MaxValue ? currentId + 1 : 0;
        }

        private void CreateTable()
        {
            try
            {
                store.AddTable(tableName);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Unable to create PushAgent store: {e.Message}.");
                throw;
            }
        }

        private Table GetTable()
        {
            return store.Tables.FirstOrDefault(t => t.BrowseName == tableName);
        }

        private void CreateColumns()
        {
            try
            {
                table.AddColumn("Id", OpcUa.DataTypes.UInt64);
                table.AddColumn("Timestamp", OpcUa.DataTypes.DateTime);
                if (logLocalTime)
                    table.AddColumn("LocalTimestamp", OpcUa.DataTypes.DateTime);

                foreach (var variableToLog in variablesToLogList)
                {
                    table.AddColumn(variableToLog.BrowseName, variableToLog.ActualDataType);

                    if (insertVariableTimestamp)
                        table.AddColumn(variableToLog.BrowseName + "_Timestamp", OpcUa.DataTypes.DateTime);

                    if (insertOpCode)
                        table.AddColumn(variableToLog.BrowseName + "_OpCode", OpcUa.DataTypes.Int32);
                }
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreDataLoggerWrapper", $"Unable to create columns of PushAgent store: {e.Message}.");
                throw;
            }
        }

        private void CreateColumnIndex(string columnName, bool unique)
        {
            string uniqueKeyWord = string.Empty;
            if (unique)
                uniqueKeyWord = "UNIQUE";
            try
            {
                string query = $"CREATE {uniqueKeyWord} INDEX \"{columnName}_index\" ON  \"{tableName}\"(\"{columnName}\")";
                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Warning("PushAgentStoreDataLoggerWrapper", $"Unable to create index on PushAgent store: {e.Message}.");
            }
        }

        private string[] GetTableColumnsOrderedByVariableName()
        {
            List<string> columnNames = new List<string>();
            columnNames.Add("Id");
            columnNames.Add("Timestamp");
            if (logLocalTime)
                columnNames.Add("LocalTimestamp");

            foreach (var variableToLog in variablesToLogList)
            {
                columnNames.Add(variableToLog.BrowseName);

                if (insertVariableTimestamp)
                    columnNames.Add(variableToLog.BrowseName + "_Timestamp");

                if (insertOpCode)
                    columnNames.Add(variableToLog.BrowseName + "_OpCode");
            }

            return columnNames.ToArray();
        }

        private string GetQuerySelectParameters()
        {
            var selectParameters = "\"Timestamp\", ";
            if (logLocalTime)
                selectParameters += "\"LocalTimestamp\", ";

            selectParameters = $"{selectParameters} {GetQueryColumnsOrderedByVariableName()}";

            return selectParameters;
        }

        private string GetQueryColumnsOrderedByVariableName()
        {
            string columnsOrderedByVariableName = string.Empty;
            foreach (var variable in variablesToLogList)
            {
                if (columnsOrderedByVariableName != string.Empty)
                    columnsOrderedByVariableName += ", ";

                columnsOrderedByVariableName += "\"" + variable.BrowseName + "\"";

                if (insertVariableTimestamp)
                    columnsOrderedByVariableName += ", \"" + variable.BrowseName + "_Timestamp\"";

                if (insertOpCode)
                    columnsOrderedByVariableName += ", \"" + variable.BrowseName + "_OpCode\"";
            }

            return columnsOrderedByVariableName;
        }

        private readonly Store store;
        private readonly Table table;
        private readonly string tableName;
        private readonly List<VariableToLog> variablesToLogList;
        private readonly string[] columns;
        private readonly bool insertOpCode;
        private readonly bool insertVariableTimestamp;
        private readonly bool logLocalTime;
        private UInt64 idCount;
    }

    public class PushAgentStoreRowPerVariableWrapper : SupportStore
    {
        public PushAgentStoreRowPerVariableWrapper(SQLiteStore store, string tableName, bool insertOpCode)
        {
            this.store = store;
            this.tableName = tableName;
            this.insertOpCode = insertOpCode;

            try
            {
                CreateTable();
                table = GetTable();
                CreateColumns();
                CreateColumnIndex("Id", true);
                CreateColumnIndex("Timestamp", false);
                columns = GetTableColumnNames();
                idCount = GetMaxId();
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Unable to create PushAgent store: {e.Message}.");
                throw;
            }
        }

        public void DeleteRecords(int numberOfRecordsToDelete)
        {
            try
            {
                string query = $"DELETE FROM \"{tableName}\" " +
                               $"ORDER BY \"Timestamp\" ASC, \"Id\" ASC " +
                               $"LIMIT {numberOfRecordsToDelete}";

                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Failed to delete data from PushAgent store: {e.Message}.");
                throw;
            }
        }

        public void InsertRecords(List<Record> records)
        {
            List<VariableRecord> variableRecords = records.Cast<VariableRecord>().ToList();
            object[,] values = new object[records.Count, columns.Length];
            UInt64 tempIdCount = idCount;
            for (int i = 0; i < variableRecords.Count; ++i)
            {
                values[i, 0] = tempIdCount;
                values[i, 1] = variableRecords[i].timestamp.Value;
                values[i, 2] = variableRecords[i].variableId;
                values[i, 3] = variableRecords[i].serializedValue;
                if (insertOpCode)
                    values[i, 4] = variableRecords[i].variableOpCode;

                tempIdCount = GetNextInternalId(tempIdCount);
            }

            try
            {
                table.Insert(columns, values);
                idCount = tempIdCount;
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Failed to insert data into PushAgent store: {e.Message}.");
                throw;
            }
        }

        public List<Record> QueryOlderEntries(int numberOfEntries)
        {
            List<VariableRecord> records = new List<VariableRecord>();
            object[,] resultSet;
            string[] header;

            try
            {
                string query = $"SELECT {GetQueryColumns()} " +
                               $"FROM \"{tableName}\" " +
                               $"ORDER BY \"Timestamp\" ASC, \"Id\" ASC " +
                               $"LIMIT {numberOfEntries}";

                store.Query(query, out header, out resultSet);

                var rowCount = resultSet != null ? resultSet.GetLength(0) : 0;
                for (int i = 0; i < rowCount; ++i)
                {
                    int? opCodeValue = (int?)null;
                    if (insertOpCode)
                    {
                        if (resultSet[i, 3] == null)
                            opCodeValue = null;
                        else
                            opCodeValue = int.Parse(resultSet[i, 3].ToString());
                    }

                    VariableRecord record;
                    if (insertOpCode)
                        record = new VariableRecord(GetTimestamp(resultSet[i, 0]),
                                                    resultSet[i, 1].ToString(),
                                                    null,
                                                    resultSet[i, 2].ToString(),
                                                    opCodeValue);
                    else
                        record = new VariableRecord(GetTimestamp(resultSet[i, 0]),
                                                    resultSet[i, 1].ToString(),
                                                    null,
                                                    resultSet[i, 2].ToString());
                    records.Add(record);
                }
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Failed to query older entries from PushAgent store: {e.Message}.");
                throw;
            }

            return records.Cast<Record>().ToList();
        }

        public long RecordsCount()
        {
            object[,] resultSet;
            long result = 0;

            try
            {
                string query = $"SELECT COUNT(*) FROM \"{tableName}\"";

                store.Query(query, out _, out resultSet);
                result = ((long)resultSet[0, 0]);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Failed to query count: {e.Message}.");
                throw;
            }

            return result;
        }

        private ulong GetMaxId()
        {
            object[,] resultSet;

            try
            {
                string query = $"SELECT MAX(\"ID\") FROM \"{tableName}\"";

                store.Query(query, out _, out resultSet);

                if (resultSet[0, 0] != null)
                    return GetNextInternalId(UInt64.Parse(resultSet[0, 0].ToString()));
                else
                    return 0;
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Failed to query maxid: {e.Message}.");
                throw;
            }
        }

        private UInt64 GetNextInternalId(UInt64 currentId)
        {
            return currentId < Int64.MaxValue ? currentId + 1 : 0;
        }

        private void CreateTable()
        {
            try
            {
                store.AddTable(tableName);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Unable to create PushAgent store: {e.Message}.");
                throw;
            }
        }

        private Table GetTable()
        {
            return store.Tables.FirstOrDefault(t => t.BrowseName == tableName);
        }

        private void CreateColumns()
        {
            try
            {
                table.AddColumn("Id", OpcUa.DataTypes.UInt64);
                table.AddColumn("Timestamp", OpcUa.DataTypes.DateTime);
                table.AddColumn("VariableId", OpcUa.DataTypes.String);
                table.AddColumn("Value", OpcUa.DataTypes.String);

                if (insertOpCode)
                    table.AddColumn("OpCode", OpcUa.DataTypes.Int32);
            }
            catch (Exception e)
            {
                Log.Error("PushAgentStoreRowPerVariableWrapper", $"Unable to create columns of PushAgent store: {e.Message}.");
                throw;
            }
        }

        private void CreateColumnIndex(string columnName, bool unique)
        {
            string uniqueKeyWord = string.Empty;
            if (unique)
                uniqueKeyWord = "UNIQUE";
            try
            {
                string query = $"CREATE {uniqueKeyWord} INDEX \"{columnName}_index\" ON  \"{tableName}\"(\"{columnName}\")";
                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Warning("PushAgentStoreRowPerVariableWrapper", $"Unable to create index on PushAgent store: {e.Message}.");
            }
        }

        private string[] GetTableColumnNames()
        {
            if (table == null)
                return null;

            var result = new List<string>();
            foreach (var column in table.Columns)
                result.Add(column.BrowseName);

            return result.ToArray();
        }

        private string GetQueryColumns()
        {
            string columns = "\"Timestamp\", ";
            columns += "\"VariableId\", ";
            columns += "\"Value\"";

            if (insertOpCode)
                columns += ", OpCode";

            return columns;
        }

        private DateTime GetTimestamp(object value)
        {
            if (Type.GetTypeCode(value.GetType()) == TypeCode.DateTime)
                return ((DateTime)value);
            else
                return DateTime.SpecifyKind(DateTime.Parse(value.ToString()), DateTimeKind.Utc);
        }

        private readonly SQLiteStore store;
        private readonly string tableName;
        private readonly Table table;
        private readonly string[] columns;
        private readonly bool insertOpCode;
        private UInt64 idCount;
    }

    public class DataLoggerStatusStoreWrapper
    {
        public DataLoggerStatusStoreWrapper(Store store,
                                            string tableName,
                                            List<VariableToLog> variablesToLogList,
                                            bool insertOpCode,
                                            bool insertVariableTimestamp)
        {
            this.store = store;
            this.tableName = tableName;
            this.variablesToLogList = variablesToLogList;
            this.insertOpCode = insertOpCode;
            this.insertVariableTimestamp = insertVariableTimestamp;

            try
            {
                CreateTable();
                table = GetTable();
                CreateColumns();
                columns = GetTableColumnsOrderedByVariableName();
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Unable to initialize internal DataLoggerStatusStoreWrapper {e.Message}.");
                throw;
            }
        }

        public void UpdateRecord(UInt64 rowId)
        {
            if (RecordsCount() == 0)
            {
                InsertRecord(rowId);
                return;
            }

            try
            {
                string query = $"UPDATE \"{tableName}\" SET \"RowId\" = {rowId} WHERE \"Id\"= 1";
                Log.Verbose1("DataLoggerStatusStoreWrapper", $"Update data logger status row id to {rowId}.");

                store.Query(query, out _, out _);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Failed to update internal data logger status: {e.Message}.");
                throw;
            }
        }

        public void InsertRecord(UInt64 rowId)
        {
            var values = new object[1, columns.Length];

            values[0, 0] = 1;
            values[0, 1] = rowId;

            try
            {
                Log.Verbose1("DataLoggerStatusStoreWrapper", $"Set data logger status row id to {rowId}.");
                table.Insert(columns, values);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Failed to update internal data logger status: {e.Message}.");
                throw;
            }
        }

        public UInt64? QueryStatus()
        {
            object[,] resultSet;
            string[] header;

            try
            {
                string query = $"SELECT \"RowId\" FROM \"{tableName}\"";

                store.Query(query, out header, out resultSet);

                if (resultSet[0, 0] != null)
                {
                    Log.Verbose1("DataLoggerStatusStoreWrapper", $"Query data logger status returns {resultSet[0, 0]}.");
                    return UInt64.Parse(resultSet[0, 0].ToString());
                }
                return null;
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Failed to query internal data logger status: {e.Message}.");
                throw;
            }
        }

        public long RecordsCount()
        {
            object[,] resultSet;
            long result = 0;

            try
            {
                string query = $"SELECT COUNT(*) FROM \"{tableName}\"";

                store.Query(query, out _, out resultSet);
                result = ((long)resultSet[0, 0]);
                Log.Verbose1("DataLoggerStatusStoreWrapper", $"Get data logger status records count returns {result}.");
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Failed to query count: {e.Message}.");
                throw;
            }

            return result;
        }

        private void CreateTable()
        {
            try
            {
                store.AddTable(tableName);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Unable to create internal table to DataLoggerStatusStore: {e.Message}.");
                throw;
            }
        }

        private Table GetTable()
        {
            return store.Tables.FirstOrDefault(t => t.BrowseName == tableName);
        }

        private void CreateColumns()
        {
            try
            {
                table.AddColumn("Id", OpcUa.DataTypes.Int32);

                // We need to store only the last query's last row's id to retrieve the dataLogger row
                table.AddColumn("RowId", OpcUa.DataTypes.Int64);
            }
            catch (Exception e)
            {
                Log.Error("DataLoggerStatusStoreWrapper", $"Unable to create columns of internal DataLoggerStatusStore: {e.Message}.");
                throw;
            }
        }

        private string[] GetTableColumnsOrderedByVariableName()
        {
            List<string> columnNames = new List<string>();
            columnNames.Add("Id");
            columnNames.Add("RowId");

            return columnNames.ToArray();
        }

        private readonly Store store;
        private readonly Table table;
        private readonly string tableName;
        private readonly List<VariableToLog> variablesToLogList;
        private readonly string[] columns;
        private readonly bool insertOpCode;
        private readonly bool insertVariableTimestamp;
    }

    public class DataLoggerRecordPuller
    {
        public DataLoggerRecordPuller(IUAObject logicObject,
                                      NodeId dataLoggerNodeId,
                                      SupportStore pushAgentStore,
                                      DataLoggerStatusStoreWrapper statusStoreWrapper,
                                      DataLoggerStoreWrapper dataLoggerStore,
                                      bool preserveDataLoggerHistory,
                                      bool pushByRow,
                                      int pullPeriod,
                                      int numberOfVariablesToLog)
        {
            this.logicObject = logicObject;
            this.pushAgentStore = pushAgentStore;
            this.statusStoreWrapper = statusStoreWrapper;
            this.dataLoggerStore = dataLoggerStore;
            this.dataLoggerNodeId = dataLoggerNodeId;
            this.preserveDataLoggerHistory = preserveDataLoggerHistory;
            this.pushByRow = pushByRow;
            this.numberOfVariablesToLog = numberOfVariablesToLog;

            if (this.preserveDataLoggerHistory)
            {
                UInt64? dataLoggerMaxId = this.dataLoggerStore.GetDataLoggerMaxId();

                if (statusStoreWrapper.RecordsCount() == 1)
                    lastPulledRecordId = statusStoreWrapper.QueryStatus();

                // Check if DataLogger has elements or if the maximum id is greater than lastPulledRecordId
                if (dataLoggerMaxId == null || (dataLoggerMaxId.HasValue && dataLoggerMaxId < lastPulledRecordId))
                    lastPulledRecordId = Int64.MaxValue;  // We have no elements in DataLogger so we will restart the count from 0
            }

            lastInsertedValues = new Dictionary<string, UAValue>();

            dataLoggerPullTask = new PeriodicTask(PullDataLoggerRecords, pullPeriod, this.logicObject);
            dataLoggerPullTask.Start();
        }

        public DataLoggerRecordPuller(IUAObject logicObject,
                                      NodeId dataLoggerNodeId,
                                      SupportStore pushAgentStore,
                                      DataLoggerStoreWrapper dataLoggerStore,
                                      bool preserveDataLoggerHistory,
                                      bool pushByRow,
                                      int pullPeriod,
                                      int numberOfVariablesToLog)
        {
            this.logicObject = logicObject;
            this.pushAgentStore = pushAgentStore;
            this.dataLoggerStore = dataLoggerStore;
            this.dataLoggerNodeId = dataLoggerNodeId;
            this.preserveDataLoggerHistory = preserveDataLoggerHistory;
            this.pushByRow = pushByRow;
            this.numberOfVariablesToLog = numberOfVariablesToLog;

            lastInsertedValues = new Dictionary<string, UAValue>();

            dataLoggerPullTask = new PeriodicTask(PullDataLoggerRecords, pullPeriod, this.logicObject);
            dataLoggerPullTask.Start();
        }

        public void StopPullTask()
        {
            dataLoggerPullTask.Cancel();
        }

        private void PullDataLoggerRecords()
        {
            try
            {
                dataLoggerPulledRecords = null;
                if (!preserveDataLoggerHistory || lastPulledRecordId == null)
                    dataLoggerPulledRecords = dataLoggerStore.QueryNewEntries();
                else
                    dataLoggerPulledRecords = dataLoggerStore.QueryNewEntriesUsingLastQueryId(lastPulledRecordId.Value);

                if (dataLoggerPulledRecords.Count > 0)
                {
                    InsertDataLoggerRecordsIntoPushAgentStore();

                    if (!preserveDataLoggerHistory)
                        dataLoggerStore.DeletePulledRecords();
                    else
                    {
                        lastPulledRecordId = dataLoggerStore.GetMaxIdFromTemporaryTable();

                        statusStoreWrapper.UpdateRecord(lastPulledRecordId.Value);
                    }

                    dataLoggerPulledRecords.Clear();
                }
            }
            catch (Exception e)
            {
                if (dataLoggerStore.GetStoreStatus() != StoreStatus.Offline)
                {
                    Log.Error("DataLoggerRecordPuller", $"Unable to retrieve data from DataLogger store: {e.Message}.");
                    StopPullTask();
                }
            }
        }

        private void InsertDataLoggerRecordsIntoPushAgentStore()
        {
            if (!IsStoreSpaceAvailable())
                return;

            if (pushByRow)
                InsertRowsIntoPushAgentStore();
            else
                InsertVariableRecordsIntoPushAgentStore();
        }

        private VariableRecord CreateVariableRecord(VariableRecord variable, DateTime recordTimestamp)
        {
            VariableRecord variableRecord;
            if (variable.timestamp == null)
                variableRecord = new VariableRecord(recordTimestamp,
                                                    variable.variableId,
                                                    variable.value,
                                                    variable.serializedValue,
                                                    variable.variableOpCode);
            else
                variableRecord = new VariableRecord(variable.timestamp,
                                                    variable.variableId,
                                                    variable.value,
                                                    variable.serializedValue,
                                                    variable.variableOpCode);



            return variableRecord;
        }

        private void InsertRowsIntoPushAgentStore()
        {
            int numberOfStorableRecords = CalculateNumberOfElementsToInsert();

            if (dataLoggerPulledRecords.Count > 0)
                pushAgentStore.InsertRecords(dataLoggerPulledRecords.Cast<Record>().ToList().GetRange(0, numberOfStorableRecords));
        }

        private void InsertVariableRecordsIntoPushAgentStore()
        {
            int numberOfStorableRecords = CalculateNumberOfElementsToInsert();

            // Temporary dictionary is used to update values, once the records are inserted then the content is copied to lastInsertedValues
            Dictionary<string, UAValue> tempLastInsertedValues = lastInsertedValues.Keys.ToDictionary(_ => _, _ => lastInsertedValues[_]);
            List<VariableRecord> pushAgentRecords = new List<VariableRecord>();
            foreach (var record in dataLoggerPulledRecords.GetRange(0, numberOfStorableRecords))
            {
                foreach (var variable in record.variables)
                {
                    VariableRecord variableRecord = CreateVariableRecord(variable, record.timestamp.Value);
                    if (GetSamplingMode() == SamplingMode.VariableChange)
                    {
                        if (!tempLastInsertedValues.ContainsKey(variable.variableId))
                        {
                            if (variableRecord.serializedValue != null)
                            {
                                pushAgentRecords.Add(variableRecord);
                                tempLastInsertedValues.Add(variableRecord.variableId, variableRecord.value);
                            }
                        }
                        else
                        {
                            if (variable.value != tempLastInsertedValues[variable.variableId] && variableRecord.serializedValue != null)
                            {
                                pushAgentRecords.Add(variableRecord);
                                tempLastInsertedValues[variableRecord.variableId] = variableRecord.value;
                            }
                        }
                    }
                    else
                    {
                        if (variableRecord.serializedValue != null)
                            pushAgentRecords.Add(variableRecord);
                    }
                }
            }

            if (pushAgentRecords.Count > 0)
            {
                pushAgentStore.InsertRecords(pushAgentRecords.Cast<Record>().ToList());

                if (GetSamplingMode() == SamplingMode.VariableChange)
                    lastInsertedValues = tempLastInsertedValues.Keys.ToDictionary(_ => _, _ => tempLastInsertedValues[_]);
            }
        }

        private int GetMaximumStoreCapacity()
        {
            return logicObject.GetVariable("MaximumStoreCapacity").Value;
        }

        private SamplingMode GetSamplingMode()
        {
            var dataLogger = InformationModel.Get<DataLogger>(dataLoggerNodeId);
            return dataLogger.SamplingMode;
        }

        private int CalculateNumberOfElementsToInsert()
        {
            // Calculate the number of records that can be effectively stored
            int numberOfStorableRecords;

            if (pushByRow)
                numberOfStorableRecords = (GetMaximumStoreCapacity() - (int)pushAgentStore.RecordsCount());
            else
            {
                if (GetSamplingMode() == SamplingMode.VariableChange)
                    numberOfStorableRecords = (GetMaximumStoreCapacity() - (int)pushAgentStore.RecordsCount());
                else
                    numberOfStorableRecords = (int)Math.Floor((double)(GetMaximumStoreCapacity() - (int)pushAgentStore.RecordsCount()) / numberOfVariablesToLog);
            }

            if (numberOfStorableRecords > dataLoggerPulledRecords.Count)
                numberOfStorableRecords = dataLoggerPulledRecords.Count;

            return numberOfStorableRecords;
        }

        private bool IsStoreSpaceAvailable()
        {
            if (pushAgentStore.RecordsCount() >= GetMaximumStoreCapacity() - 1)
            {
                Log.Warning("DataLoggerRecordPuller", "Maximum store capacity reached! Skipping...");
                return false;
            }

            var percentageStoreCapacity = ((double)pushAgentStore.RecordsCount() / GetMaximumStoreCapacity()) * 100;
            if (percentageStoreCapacity >= 70)
                Log.Warning("DataLoggerRecordPuller", "Store capacity 70% reached!");

            return true;
        }

        private List<DataLoggerRecord> dataLoggerPulledRecords;
        private UInt64? lastPulledRecordId;
        private readonly PeriodicTask dataLoggerPullTask;
        private readonly SupportStore pushAgentStore;
        private readonly DataLoggerStatusStoreWrapper statusStoreWrapper;
        private readonly DataLoggerStoreWrapper dataLoggerStore;
        private readonly bool preserveDataLoggerHistory;
        private readonly bool pushByRow;
        private readonly IUAObject logicObject;
        private readonly int numberOfVariablesToLog;
        private readonly NodeId dataLoggerNodeId;
        private Dictionary<string, UAValue> lastInsertedValues;
    }

    public class MQTTConnector : IDisposable
    {
        private const int MQTT_BROKER_DEFAULT_PORT = 1883;
        private const int MQTT_BROKER_DEFAULT_SSL_PORT = 8883;

        public MQTTConnector(string brokerIpAddressVariable,
                             string clientID,
                             CancellationToken cancellationToken,
                             int port,
                             bool useTLS,
                             string pathCACert,
                             string pathClientCert,
                             string passwordClientCert,
                             string username,
                             string password)
        {
            this.brokerIpAddress = brokerIpAddressVariable;

            if (port == 0 && useTLS)
            {
                this.brokerPort = MQTT_BROKER_DEFAULT_SSL_PORT;
            }
            else if (port == 0 && !useTLS)
            {
                this.brokerPort = MQTT_BROKER_DEFAULT_PORT;
            }
            else
            {
                this.brokerPort = port;
            }

            this.clientID = clientID;
            this.cancellationToken = cancellationToken;

            var factory = new MqttFactory();
            mqttClient = factory.CreateMqttClient();

            mqttClient.InspectPacketAsync += InspectPacketAsync;
            mqttClient.DisconnectedAsync += ConnectionClosedHandlerAsync;
            connectEvent = new AutoResetEvent(false);

            if (useTLS)
            {
                tlsParameters = new MqttClientOptionsBuilderTlsParameters()
                {
                    UseTls = true
                };

                if (!String.IsNullOrWhiteSpace(pathClientCert))
                {
                    var clientCert = new X509Certificate2(pathClientCert, passwordClientCert);
                    List<X509Certificate> certificates = new List<X509Certificate>() { clientCert };
                    tlsParameters.Certificates = certificates;
                }
                if (!String.IsNullOrWhiteSpace(pathCACert))
                {
                    this.pathCACert = pathCACert;

                    //set certificate validation callback for CA not from OS certificate store
                    tlsParameters.CertificateValidationHandler = RemoteCertificateValidationCallback;
                }
            }

            if (!string.IsNullOrWhiteSpace(username))
            {
                this.username = username;
                this.password = password;
                this.useUsernamePassword = true;
            }
        }

        public void Dispose()
        {
            mqttClient.InspectPacketAsync -= InspectPacketAsync;
            mqttClient.DisconnectedAsync -= ConnectionClosedHandlerAsync;
            mqttClient.Dispose();
            GC.SuppressFinalize(this);
        }

        public async Task ConnectAsync()
        {
            try
            {
                cancellationToken.ThrowIfCancellationRequested();
                Log.Verbose1("MQTTConnector", $"Connecting to MQTT broker ({brokerIpAddress}) on port {brokerPort} with client ID ({clientID}).");
                var optionsBuilder = new MqttClientOptionsBuilder()
                                .WithTcpServer(brokerIpAddress, brokerPort)
                                .WithClientId(clientID)
                                .WithKeepAlivePeriod(new TimeSpan(0, 0, keepAlivePeriod))
                                .WithRequestProblemInformation();

                if (useUsernamePassword)
                {
                    Log.Verbose1("MQTTConnector", $"Connecting with username: {username}.");
                    optionsBuilder.WithCredentials(username, password);
                }

                if (tlsParameters != null)
                {
                    optionsBuilder.WithTls(tlsParameters);
                }

                MqttClientConnectResult result = await mqttClient.ConnectAsync(optionsBuilder.Build(), cancellationToken).ConfigureAwait(false);

                if (result.ResultCode != MqttClientConnectResultCode.Success)
                {
                    throw new CoreException($"Connection failed. Code: {result.ResultCode}. Reason: {result.ReasonString}).");
                }

                connectEvent?.Set();
                numberOfRetries = 0;

                Log.Verbose1("MQTTConnector", "Connection completed.");
            }
            catch (OperationCanceledException)
            {
                connectEvent?.Set();
            }
            catch (Exception e)
            {
                Log.Error("MQTTConnector", $"Error occurred when connecting to MQTT broker: {e.Message}");
                throw;
            }
        }

        public async void DisconnectAsync()
        {
            const int timeout = 10000; //ms

            try
            {
                if (mqttClient.IsConnected)
                {
                    Log.Verbose1("MQTTConnector", $"Disconnecting from MQTT broker ({brokerIpAddress}) on port {brokerPort} with client ID ({clientID}).");
                    await mqttClient.DisconnectAsync(cancellationToken: new CancellationTokenSource(timeout).Token)
                        .ConfigureAwait(false);
                    Log.Verbose1("MQTTConnector", "Disconnection completed.");
                }
            }
            catch (OperationCanceledException)
            {
                Log.Warning("MQTTConnector", "Disconnection from MQTT broker timed out.");
            }
            catch (Exception e)
            {
                Log.Error("MQTTConnector", $"Error occurred while disconnecting from MQTT broker: {e.Message}");
                throw;
            }
        }

        private Task InspectPacketAsync(MQTTnet.Diagnostics.InspectMqttPacketEventArgs arg)
        {
            // Do nothing
            return Task.CompletedTask;
        }

        private bool RemoteCertificateValidationCallback(MqttClientCertificateValidationEventArgs arg)
        {
            if (arg.Certificate != null)
            {
                Log.Verbose1("MQTTConnector", $"Remote certificate validation, Issuer: {arg.Certificate.Issuer}, Subject: {arg.Certificate.Subject}.");
            }

            if (arg.SslPolicyErrors.HasFlag(System.Net.Security.SslPolicyErrors.RemoteCertificateNameMismatch))
            {
                Log.Warning("PushAgent", "Remote certificate name mismatch");
                return false;
            }

            bool result = false;
            try
            {
                // Load the ca.crt
                X509Certificate2 caCrt = new X509Certificate2(File.ReadAllBytes(this.pathCACert));

                using (X509Chain chain = new X509Chain())
                {
                    chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
                    chain.ChainPolicy.RevocationFlag = X509RevocationFlag.ExcludeRoot;
                    chain.ChainPolicy.VerificationFlags = X509VerificationFlags.NoFlag;
                    chain.ChainPolicy.VerificationTime = DateTime.Now;
                    chain.ChainPolicy.UrlRetrievalTimeout = TimeSpan.Zero;
                    chain.ChainPolicy.CustomTrustStore.Add(caCrt);
                    chain.ChainPolicy.TrustMode = X509ChainTrustMode.CustomRootTrust;

                    // convert provided X509Certificate to X509Certificate2
                    var x5092 = new X509Certificate2(arg.Certificate);
                    result = chain.Build(x5092);

                    foreach (var problem in chain.ChainStatus)
                    {
                        Log.Warning("PushAgent", $"Certificate chain problem found: {problem.Status} - {problem.StatusInformation}");
                    }
                }
            }
            catch (Exception e)
            {
                Log.Verbose1("MQTTConnector", $"Certificate validation failed ({e.Message}).");
                return false;
            }

            if (!result)
            {
                Log.Verbose1("PushAgent", "Remote certificate validation failed");
            }

            return result;
        }

        private int CalculateNextRetryTimeout()
        {
            var retryTimeoutMs = Math.Pow(2, numberOfRetries) * initialRetryTimeoutMs;
            retryTimeoutMs = Math.Min(retryTimeoutMs, maximumRetryTimeoutMs);

            return (int)retryTimeoutMs;
        }

        private async Task ConnectionClosedHandlerAsync(MqttClientDisconnectedEventArgs arg)
        {
            if (!cancellationToken.IsCancellationRequested)
            {
                if (arg.ConnectResult != null)
                    Log.Error("MQTTConnector", $"Connection closed. Code: {arg.ConnectResult.ResultCode}. Reason: {arg.ConnectResult.ReasonString}.");

                Interlocked.Increment(ref numberOfRetries);

                var retryTimeoutMs = CalculateNextRetryTimeout();
                Log.Info("MQTTConnector", $"Next connection attempt in {retryTimeoutMs} ms (attempt {numberOfRetries}).");
                try
                {
                    await Task.Delay(retryTimeoutMs, cancellationToken).ConfigureAwait(false);
                }
                finally
                {
                    await ConnectAsync().ConfigureAwait(false);
                }
            }
        }

        public async Task PublishAsync(string records, string topic, bool retain, int qosLevel)
        {
            if (!mqttClient.IsConnected)
            {
                connectEvent.WaitOne();
            }

            cancellationToken.ThrowIfCancellationRequested();
            Log.Verbose1("MQTTConnector", $"Publishing {records} to MQTT broker on {topic} topic.");

            MqttClientPublishResult result = await mqttClient.PublishBinaryAsync(topic,
                                                                                 Encoding.UTF8.GetBytes(records), // message body
                                                                                 GetQoSLevel(qosLevel),
                                                                                 retain,
                                                                                 cancellationToken)
                                                             .ConfigureAwait(false);

            if (!result.IsSuccess)
            {
                throw new CoreException($"Publish failed. Code: {result.ReasonCode}. Reason: {result.ReasonString}).");
            }
        }

        private static MQTTnet.Protocol.MqttQualityOfServiceLevel GetQoSLevel(int qosLevel)
        {
            switch (qosLevel)
            {
                case 0:
                    return MQTTnet.Protocol.MqttQualityOfServiceLevel.AtMostOnce;
                case 1:
                    return MQTTnet.Protocol.MqttQualityOfServiceLevel.AtLeastOnce;
                case 2:
                    return MQTTnet.Protocol.MqttQualityOfServiceLevel.ExactlyOnce;
                default:
                    return MQTTnet.Protocol.MqttQualityOfServiceLevel.AtLeastOnce;
            }
        }

        private readonly IMqttClient mqttClient;
        private readonly string clientID;
        private readonly string brokerIpAddress;
        private readonly int brokerPort;
        private readonly MqttClientOptionsBuilderTlsParameters tlsParameters;
        private AutoResetEvent connectEvent;
        private int numberOfRetries = 0;
        private int initialRetryTimeoutMs = 1000;
        private int maximumRetryTimeoutMs = 60000;
        private CancellationToken cancellationToken;
        private readonly ushort keepAlivePeriod = 5; //seconds
        private readonly bool useUsernamePassword;
        private readonly string username;
        private readonly string password;
        private readonly string pathCACert;
    }

    public class JSONBuilder
    {
        public JSONBuilder(bool insertOpCode, bool insertVariableTimestamp, bool logLocalTime)
        {
            this.insertOpCode = insertOpCode;
            this.insertVariableTimestamp = insertVariableTimestamp;
            this.logLocalTime = logLocalTime;
        }

        public string CreatePacketFormatJSON(DataLoggerRowPacket packet)
        {
            var sb = new StringBuilder();
            var sw = new StringWriter(sb);
            using (var writer = new JsonTextWriter(sw))
            {

                writer.Formatting = Formatting.None;

                writer.WriteStartObject();
                writer.WritePropertyName("Timestamp");
                writer.WriteValue(packet.timestamp);
                writer.WritePropertyName("ClientId");
                writer.WriteValue(packet.clientId);
                writer.WritePropertyName("Rows");
                writer.WriteStartArray();
                foreach (var record in packet.records)
                {
                    writer.WriteStartObject();
                    writer.WritePropertyName("RowTimestamp");
                    writer.WriteValue(record.timestamp);

                    if (logLocalTime)
                    {
                        writer.WritePropertyName("RowLocalTimestamp");
                        writer.WriteValue(record.localTimestamp);
                    }

                    writer.WritePropertyName("Variables");
                    writer.WriteStartArray();
                    foreach (var variable in record.variables)
                    {
                        writer.WriteStartObject();

                        writer.WritePropertyName("VariableName");
                        writer.WriteValue(variable.variableId);
                        writer.WritePropertyName("Value");
                        writer.WriteValue(variable.value?.Value);

                        if (insertVariableTimestamp)
                        {
                            writer.WritePropertyName("VariableTimestamp");
                            writer.WriteValue(variable.timestamp);
                        }

                        if (insertOpCode)
                        {
                            writer.WritePropertyName("VariableOpCode");
                            writer.WriteValue(variable.variableOpCode);
                        }

                        writer.WriteEndObject();
                    }
                    writer.WriteEnd();
                    writer.WriteEndObject();
                }
                writer.WriteEnd();
                writer.WriteEndObject();
            }

            return sb.ToString();
        }

        public string CreatePacketFormatJSON(VariablePacket packet)
        {
            var sb = new StringBuilder();
            var sw = new StringWriter(sb);
            using (var writer = new JsonTextWriter(sw))
            {
                writer.Formatting = Formatting.None;

                writer.WriteStartObject();
                writer.WritePropertyName("Timestamp");
                writer.WriteValue(packet.timestamp);
                writer.WritePropertyName("ClientId");
                writer.WriteValue(packet.clientId);
                writer.WritePropertyName("Records");
                writer.WriteStartArray();
                foreach (var record in packet.records)
                {
                    writer.WriteStartObject();

                    writer.WritePropertyName("VariableName");
                    writer.WriteValue(record.variableId);
                    writer.WritePropertyName("SerializedValue");
                    writer.WriteValue(record.serializedValue);
                    writer.WritePropertyName("VariableTimestamp");
                    writer.WriteValue(record.timestamp);

                    if (insertOpCode)
                    {
                        writer.WritePropertyName("VariableOpCode");
                        writer.WriteValue(record.variableOpCode);
                    }

                    writer.WriteEndObject();
                }
                writer.WriteEnd();
                writer.WriteEndObject();
            }

            return sb.ToString();
        }

        private readonly bool insertOpCode;
        private readonly bool insertVariableTimestamp;
        private readonly bool logLocalTime;
    }
}

public class PushAgent : BaseNetLogic
{
    public override void Start()
    {
        Log.Verbose1("PushAgent", "Start push agent.");
        CultureInfo.DefaultThreadCurrentCulture = new CultureInfo("en-US");
        try
        {
            cancellationTokenSource = new CancellationTokenSource();
            LoadPushAgentConfiguration();
            CheckMQTTParameters();

            ConfigureStores();
            ConfigureDataLoggerRecordPuller();
            ConfigureMQTT();

            mqttClientConnector.ConnectAsync();

            StartFetchTimer();
        }
        catch (Exception e)
        {
            Log.Error("PushAgent", $"Unable to initialize PushAgent, an error occurred: {e.Message}.");
            throw;
        }
    }

    public override void Stop()
    {
        Log.Verbose1("PushAgent", "Stop push agent.");
        cancellationTokenSource.Cancel();

        try
        {
            dataLoggerRecordPuller.StopPullTask();
            lock (dataFetchLock)
            {
                dataFetchTask.Cancel();
            }
        }
        catch (Exception e)
        {
            Log.Warning("PushAgent", $"Error occurred during stoping push agent: {e.Message}");
        }
        finally
        {
            mqttClientConnector?.DisconnectAsync();
            mqttClientConnector?.Dispose();
        }
    }

    private void ConfigureMQTT()
    {
        mqttClientConnector = new MQTTConnector(pushAgentConfigurationParameters.mqttConfigurationParameters.brokerIPAddress,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.clientId,
                                                cancellationTokenSource.Token,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.brokerPort,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.useSSL,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.pathCACert,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.pathClientCert,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.passwordClientCert,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.username,
                                                pushAgentConfigurationParameters.mqttConfigurationParameters.password);
    }

    private void ConfigureDataLoggerRecordPuller()
    {
        int dataLoggerPullPeriod = LogicObject.GetVariable("DataLoggerPullTime").Value; // Period used to pull new data from the DataLogger

        if (pushAgentConfigurationParameters.preserveDataLoggerHistory)
        {
            dataLoggerRecordPuller = new DataLoggerRecordPuller(LogicObject,
                                                                LogicObject.GetVariable("DataLogger").Value,
                                                                pushAgentStore,
                                                                statusStoreWrapper,
                                                                dataLoggerStore,
                                                                pushAgentConfigurationParameters.preserveDataLoggerHistory,
                                                                pushAgentConfigurationParameters.pushFullSample,
                                                                dataLoggerPullPeriod,
                                                                pushAgentConfigurationParameters.dataLogger.VariablesToLog.ToList().Count);
        }
        else
        {
            dataLoggerRecordPuller = new DataLoggerRecordPuller(LogicObject,
                                                                LogicObject.GetVariable("DataLogger").Value,
                                                                pushAgentStore,
                                                                dataLoggerStore,
                                                                pushAgentConfigurationParameters.preserveDataLoggerHistory,
                                                                pushAgentConfigurationParameters.pushFullSample,
                                                                dataLoggerPullPeriod,
                                                                pushAgentConfigurationParameters.dataLogger.VariablesToLog.ToList().Count);
        }
    }

    private void ConfigureStores()
    {
        string pushAgentStoreBrowseName = "PushAgentStore";
        string pushAgentFilename = "push_agent_store";
        CreatePushAgentStore(pushAgentStoreBrowseName, pushAgentFilename);

        var variableLogOpCode = pushAgentConfigurationParameters.dataLogger.GetVariable("LogVariableOperationCode");
        insertOpCode = variableLogOpCode != null ? (bool)variableLogOpCode.Value : false;

        var variableTimestamp = pushAgentConfigurationParameters.dataLogger.GetVariable("LogVariableTimestamp");
        insertVariableTimestamp = variableTimestamp != null ? (bool)variableTimestamp.Value : false;

        var logLocalTimestamp = pushAgentConfigurationParameters.dataLogger.GetVariable("LogLocalTime");
        logLocalTime = logLocalTimestamp != null ? (bool)logLocalTimestamp.Value : false;

        jsonCreator = new JSONBuilder(insertOpCode, insertVariableTimestamp, logLocalTime);

        dataLoggerStore = new DataLoggerStoreWrapper(InformationModel.Get<FTOptix.Store.Store>(pushAgentConfigurationParameters.dataLogger.Store),
                                            GetDataLoggerTableName(),
                                            pushAgentConfigurationParameters.dataLogger.VariablesToLog.ToList(),
                                            insertOpCode,
                                            insertVariableTimestamp,
                                            logLocalTime);

        if (!pushAgentConfigurationParameters.pushFullSample)
        {
            string tableName = "PushAgentTableRowPerVariable";
            pushAgentStore = new PushAgentStoreRowPerVariableWrapper(LogicObject.Get<SQLiteStore>(pushAgentStoreBrowseName),
                                                                     tableName,
                                                                     insertOpCode);
        }
        else
        {
            string tableName = "PushAgentTableDataLogger";
            pushAgentStore = new PushAgentStoreDataLoggerWrapper(LogicObject.Get<SQLiteStore>(pushAgentStoreBrowseName),
                                                                tableName,
                                                                pushAgentConfigurationParameters.dataLogger.VariablesToLog.ToList(),
                                                                insertOpCode,
                                                                insertVariableTimestamp,
                                                                logLocalTime);
            if (GetMaximumRecordsPerPacket() != 1)
            {
                Log.Warning("PushAgent", "For PushByRow mode maximum one row per packet is supported. Setting value to 1.");
                LogicObject.GetVariable("MaximumItemsPerPacket").Value = 1;
            }
        }

        if (pushAgentConfigurationParameters.preserveDataLoggerHistory)
        {
            string tableName = "DataLoggerStatusStore";
            statusStoreWrapper = new DataLoggerStatusStoreWrapper(LogicObject.Get<SQLiteStore>(pushAgentStoreBrowseName),
                                                                                            tableName,
                                                                                            pushAgentConfigurationParameters.dataLogger.VariablesToLog.ToList(),
                                                                                            insertOpCode,
                                                                                            insertVariableTimestamp);
        }
    }

    private void StartFetchTimer()
    {
        if (cancellationTokenSource.IsCancellationRequested)
            return;
        try
        {
            // Set the correct timeout by checking number of records to be sent
            if (pushAgentStore.RecordsCount() >= GetMaximumRecordsPerPacket())
                nextRestartTimeout = GetMinimumPublishTime();
            else
                nextRestartTimeout = GetMaximumPublishTime();
            dataFetchTask = new DelayedTask(OnFetchRequired, nextRestartTimeout, LogicObject);

            lock (dataFetchLock)
            {
                dataFetchTask.Start();
            }
            Log.Verbose1("PushAgent", $"Fetching next data in {nextRestartTimeout} ms.");
        }
        catch (Exception e)
        {
            OnFetchError(e.Message);
        }
    }

    private void OnFetchRequired()
    {
        if (pushAgentStore.RecordsCount() > 0)
            FetchData();
        else
            StartFetchTimer();
    }

    private void FetchData()
    {
        if (cancellationTokenSource.IsCancellationRequested)
            return;

        Log.Verbose1("PushAgent", "Fetching data from push agent temporary store");
        var records = GetRecordsToSend();

        if (records.Count > 0)
        {
            Publish(GenerateJSON(records));
        }
    }

    private List<Record> GetRecordsToSend()
    {
        List<Record> result = null;
        try
        {
            result = pushAgentStore.QueryOlderEntries(GetMaximumRecordsPerPacket());
        }
        catch (Exception e)
        {
            OnFetchError(e.Message);
        }
        return result;
    }

    private string GenerateJSON(List<Record> records)
    {
        var now = DateTime.Now;
        var clientId = pushAgentConfigurationParameters.mqttConfigurationParameters.clientId;

        if (pushAgentConfigurationParameters.pushFullSample)
        {
            pendingSendPacket = new DataLoggerRowPacket(now, clientId, records.Cast<DataLoggerRecord>().ToList());
            return jsonCreator.CreatePacketFormatJSON((DataLoggerRowPacket)pendingSendPacket);
        }
        else
        {
            pendingSendPacket = new VariablePacket(now, clientId, records.Cast<VariableRecord>().ToList());
            return jsonCreator.CreatePacketFormatJSON((VariablePacket)pendingSendPacket);
        }
    }

    private void Publish(string json)
    {
        try
        {
            mqttClientConnector.PublishAsync(json,
                                             pushAgentConfigurationParameters.mqttConfigurationParameters.brokerTopic,
                                             false,
                                             pushAgentConfigurationParameters.mqttConfigurationParameters.qos)
                .Wait();
            DeleteRecordsFromTempStore();
            StartFetchTimer();
        }
        catch (OperationCanceledException)
        {
            // empty
        }
        catch (Exception e)
        {
            Log.Warning("PushAgent", $"Error occurred during publishing: {e.Message}");
            StartFetchTimer();
        }
    }

    private void DeleteRecordsFromTempStore()
    {
        try
        {
            Log.Verbose1("PushAgent", "Delete records from push agent temporary store.");
            if (pushAgentConfigurationParameters.pushFullSample)
                pushAgentStore.DeleteRecords(((DataLoggerRowPacket)pendingSendPacket).records.Count);
            else
                pushAgentStore.DeleteRecords(((VariablePacket)pendingSendPacket).records.Count);
            pendingSendPacket = null;
        }
        catch (Exception e)
        {
            OnFetchError(e.Message);
        }
    }

    private void OnFetchError(string message)
    {
        Log.Error("PushAgent", $"Error while fetching data: {message}.");
        dataLoggerRecordPuller.StopPullTask();
        lock (dataFetchLock)
        {
            dataFetchTask.Cancel();
        }
    }

    private void LoadMQTTConfiguration()
    {
        pushAgentConfigurationParameters.mqttConfigurationParameters = new MQTTConfigurationParameters
        {
            clientId = LogicObject.GetVariable("ClientId").Value,
            brokerIPAddress = LogicObject.GetVariable("BrokerIPAddress").Value,
            brokerPort = LogicObject.GetVariable("BrokerPort").Value,
            brokerTopic = LogicObject.GetVariable("BrokerTopic").Value,
            qos = LogicObject.GetVariable("QoS").Value,
            useSSL = LogicObject.GetVariable("UseSSL").Value,
            pathCACert = ResourceUriValueToAbsoluteFilePath(LogicObject.GetVariable("UseSSL/CACert").Value),
            pathClientCert = ResourceUriValueToAbsoluteFilePath(LogicObject.GetVariable("UseSSL/ClientCert").Value),
            passwordClientCert = LogicObject.GetVariable("UseSSL/ClientCertPassword").Value,
            username = LogicObject.GetVariable("Username").Value,
            password = LogicObject.GetVariable("Password").Value
        };
    }

    private void LoadPushAgentConfiguration()
    {
        pushAgentConfigurationParameters = new PushAgentConfigurationParameters();

        try
        {
            LoadMQTTConfiguration();

            pushAgentConfigurationParameters.dataLogger = GetDataLogger();
            pushAgentConfigurationParameters.pushFullSample = LogicObject.GetVariable("PushFullSample").Value;
            pushAgentConfigurationParameters.preserveDataLoggerHistory = LogicObject.GetVariable("PreserveDataLoggerHistory").Value;
        }
        catch (Exception e)
        {
            throw new CoreConfigurationException("PushAgent: Configuration error", e);
        }

    }

    private void CheckMQTTParameters()
    {
        if (pushAgentConfigurationParameters.mqttConfigurationParameters.useSSL && string.IsNullOrWhiteSpace(pushAgentConfigurationParameters.mqttConfigurationParameters.pathCACert))
        {
            Log.Warning("PushAgent", "CA certificate path is not set. Set CA certificate path or install CA certificate in the system.");
        }
        var qos = pushAgentConfigurationParameters.mqttConfigurationParameters.qos;
        if (qos < 0 || qos > 2)
        {
            Log.Warning("PushAgent", "QoS Values valid are 0, 1, 2.");
        }
    }

    private int GetMaximumRecordsPerPacket()
    {
        return LogicObject.GetVariable("MaximumItemsPerPacket").Value;
    }

    private int GetMaximumPublishTime()
    {
        return LogicObject.GetVariable("MaximumPublishTime").Value;
    }

    private int GetMinimumPublishTime()
    {
        return LogicObject.GetVariable("MinimumPublishTime").Value;
    }

    private DataLogger GetDataLogger()
    {
        var dataLoggerNodeId = LogicObject.GetVariable("DataLogger").Value;
        return InformationModel.Get<DataLogger>(dataLoggerNodeId);
    }

    private string ResourceUriValueToAbsoluteFilePath(UAValue value)
    {
        var resourceUri = new ResourceUri(value);
        return resourceUri.Uri;
    }

    private string GetDataLoggerTableName()
    {
        if (pushAgentConfigurationParameters.dataLogger.TableName != null)
            return pushAgentConfigurationParameters.dataLogger.TableName;

        return pushAgentConfigurationParameters.dataLogger.BrowseName;
    }

    private void CreatePushAgentStore(string browsename, string filename)
    {
        Log.Verbose1("PushAgent", $"Create push agent store with filename: {filename}.");
        try
        {
            SQLiteStore store = InformationModel.MakeObject<SQLiteStore>(browsename);
            store.Filename = filename;
            LogicObject.Add(store);
        }
        catch (Exception e)
        {
            Log.Error("PushAgent", $"Unable to create push agent store ({e.Message}).");
            throw;
        }
    }

    private readonly object dataFetchLock = new object();
    private bool insertOpCode;
    private bool insertVariableTimestamp;
    private bool logLocalTime;
    private int nextRestartTimeout;
    private Packet pendingSendPacket;
    private DelayedTask dataFetchTask;
    private PushAgentConfigurationParameters pushAgentConfigurationParameters;
    private MQTTConnector mqttClientConnector;
    private SupportStore pushAgentStore;
    private DataLoggerStoreWrapper dataLoggerStore;
    private DataLoggerStatusStoreWrapper statusStoreWrapper;
    private JSONBuilder jsonCreator;
    private CancellationTokenSource cancellationTokenSource;
    DataLoggerRecordPuller dataLoggerRecordPuller;

    class MQTTConfigurationParameters
    {
        public string clientId;
        public string brokerIPAddress;
        public int brokerPort;
        public string brokerTopic;
        public int qos;
        public bool useSSL;
        public string pathClientCert;
        public string passwordClientCert;
        public string pathCACert;
        public string username;
        public string password;
    }

    class PushAgentConfigurationParameters
    {
        public MQTTConfigurationParameters mqttConfigurationParameters;
        public DataLogger dataLogger;
        public bool pushFullSample;
        public bool preserveDataLoggerHistory;
    }
}
