#region Using directives
using System;
using UAManagedCore;
using OpcUa = UAManagedCore.OpcUa;
using FTOptix.OPCUAServer;
using FTOptix.HMIProject;
using FTOptix.UI;
using FTOptix.NativeUI;
using FTOptix.CoreBase;
using FTOptix.Core;
using FTOptix.NetLogic;
#endregion

using System.IO;
using ICSharpCode.SharpZipLib.Zip;
using ICSharpCode.SharpZipLib.Core;

public class ZipUnzipLogic : BaseNetLogic
{
    [ExportMethod]
    public void CreateZipArchive()
    {
        var sourceValue = GetNetLogicVariableValue("Source");
        if (sourceValue == null)
            return;

        var source = new ResourceUri(sourceValue).Uri;
        var targetValue = GetNetLogicVariableValue("Target");

        if (targetValue == null)
            return;

        var target = new ResourceUri(targetValue).Uri;
        var targetFileName = SanitizeFileName(target);

        var password = GetNetLogicVariableValue("Password");
        if (password == null)
            return;

        Log.Info("ZipUnzipLogic", $"Creating Zip {targetFileName}");
        try
        {
            using (FileStream fsout = File.Create(targetFileName))
            {
                using (var zipStream = new ZipOutputStream(fsout))
                {
                    zipStream.SetLevel(9);
                    zipStream.Password = password;
                    if (File.Exists(source))
                    {
                        string zipEntryName = Path.GetFileName(source);
                        CompressFile(source, zipEntryName, zipStream);
                    }
                    else
                    {
                        CompressFolder(source, source, zipStream);
                    }
                }
            }

            Log.Info("ZipUnzipLogic", $"Zip created successfully");
        }
        catch (Exception e)
        {
            Log.Error("ZipUnzipLogic", $"Could not create zip: {e.Message}");
        }
    }

    [ExportMethod]
    public void UnzipArchive()
    {
        var sourceValue = GetNetLogicVariableValue("Source");
        if (sourceValue == null)
            return;

        var source = new ResourceUri(sourceValue).Uri;
        var targetValue = GetNetLogicVariableValue("Target");
        if (targetValue == null)
            return;

        var target = new ResourceUri(targetValue).Uri;
        var password = GetNetLogicVariableValue("Password");

        if (!File.Exists(source))
        {
            Log.Error("ZipUnzipLogic", $"Could not Unzip: File {source} does not exist");
            return;
        }

        try
        {
            using (Stream fileInputStream = File.OpenRead(source))
            {
                using (var zipFile = new ZipFile(fileInputStream))
                {
                    ExtractZipEntriesFrom(zipFile, password, target);
                }
            }

            Log.Info("ZipUnzipLogic", $"Zip successfully extracted to {target}");
        }
        catch (Exception e)
        {
            Log.Error("ZipUnzipLogic", $"Could not extract zip {e.Message}");
        }
    }

    private void ExtractZipEntriesFrom(ZipFile zipFile, string password, string targetFile)
    {
        if (!string.IsNullOrEmpty(password))
            zipFile.Password = password; 

        foreach (ZipEntry zipEntry in zipFile)
            CopyZipEntryToTargetFolder(zipFile, zipEntry, targetFile);
    }

    private void CompressFolder(string directoryRootPath, string directoryPath, ZipOutputStream zipStream)
    {
        var files = Directory.GetFiles(directoryPath);
        foreach (var file in files)
        {
            string zipEntryName = GetFileRelativePath(directoryRootPath, file);
            CompressFile(file, zipEntryName, zipStream);
        }

        //Recursively compress sub directories
        var directories = Directory.GetDirectories(directoryPath);
        foreach (var directoryName in directories)
            CompressFolder(directoryRootPath, directoryName, zipStream);
    }

    private void CompressFile(string fileName, string zipEntryName, ZipOutputStream zipStream)
    {
        try
        {
            zipStream.PutNextEntry(new ZipEntry(ZipEntry.CleanName(zipEntryName)));
            using (var fsInput = File.OpenRead(fileName))
                StreamUtils.Copy(fsInput, zipStream, bufferSize);
        }
        catch (Exception e)
        {
            Log.Error("ZipUnzipLogic", $"Could not copy file entry to zip: {e.Message}");
            throw;
        }
    }

    private string GetFileRelativePath(string rootDir, string sourceFileName)
    {
        int folderOffset = rootDir.Length;
        return sourceFileName.Substring(folderOffset).Trim('\\');
    }

    private void CopyZipEntryToTargetFolder(ZipFile zipFile, ZipEntry zipEntry, string target)
    {
        var fullFileOutputPath = Path.Combine(target, zipEntry.Name);
        CreateOutputDirectory(fullFileOutputPath);

        try
        {
            using (var zipStream = zipFile.GetInputStream(zipEntry))
            {
                using (Stream fsOutput = File.Create(fullFileOutputPath))
                {
                    StreamUtils.Copy(zipStream, fsOutput, bufferSize);
                }
            }
        }
        catch (Exception e)
        {
            Log.Error("ZipUnzipLogic", $"Could not copy zip entry to output folder: {e.Message}");
            throw;
        }
    }

    private void CreateOutputDirectory(string targetDirectoryPath)
    {
        var directoryName = Path.GetDirectoryName(targetDirectoryPath);
        if (directoryName.Length > 0)
            Directory.CreateDirectory(directoryName);
    }

    private string SanitizeFileName(string fileName)
    {
        if (fileName.EndsWith(".zip"))
            return fileName;
        else
            return $"{fileName}.zip";
    }

    private UAValue GetNetLogicVariableValue(string variableName)
    {
        var netlogicVariable = LogicObject.GetVariable(variableName);
        if (netlogicVariable == null)
        {
            Log.Error("ZipUnzipLogic", $"Could not find variable {variableName}");
            return null;
        }

        var netlogicVariableValue = netlogicVariable.Value;
        if (netlogicVariableValue == null)
        {
            Log.Error("ZipUnzipLogic", $"Could not find variable {variableName} value");
            return null;
        }
        return netlogicVariableValue;
    }

    private readonly byte[] bufferSize = new byte[4096];
}
